/* -*-C++-*-
 * ###################################################################
 *	Cpptcl - integrating C++ with Tcl
 * 
 *	FILE: "cpplist.h"
 *									  created: 24/12/95 {2:20:12 am} 
 *								  last update: 05/22/98 {10:15:31 AM} 
 *	Author:	Vince Darley
 *	E-mail:	<darley@fas.harvard.edu>
 *	  mail:	Division of	Applied	Sciences, Harvard University
 *			Oxford Street, Cambridge MA	02138, USA
 *	   www:	<http://www.fas.harvard.edu/~darley/>
 *	
 *	Description: In-line wrapper for the void* list_v class.  Allows
 *	  memory efficient multiple instantiations of pointer classes
 *	  with tiny overhead.
 * 
 *	History
 * 
 *	modified by	 rev reason
 *	-------- --- --- -----------
 *	30/1/97	 VMD 1.0 original
 * ###################################################################
 */


#ifndef _cpptcl_list_
#define _cpptcl_list_

#include "list_v.h"



// Some forward references.  We're defining three
// mutually-using classes.
template<class T>
class list;
 
template<class T> 
class list_pos;


template<class T> struct list_node;

//template<class T> ostream&  operator << (ostream&, const list<T>&);

template<class T> 
struct list_node: public list_node_v {
  private:
	friend class list<T>;
	friend class list_pos<T>;
	//friend class ostream& operator <<(ostream&, const list<T>&);
	T& t() { return (T&)_t;}
	list_node<T>& next() { return *(list_node<T>*)_next;}
	list_node<T>& prev() { return *(list_node<T>*)_prev;}
	list_node(list_node<T> *prev, const T tt,  list_node<T> *next)
	  :list_node_v(prev,tt,next) {}
	~list_node(void) {} // Just delete this node
};

// --------------------  list --------------------

template<class T> 
class list: public list_v {
  public:

	// Create lists with 0-4 elements....
	list ():list_v() {}
	list (T t1):list_v() {append(t1);}
	list (T t1, T t2):list_v() {append(t1);append(t2);} 
	list (T t1, T t2, T t3):list_v() {
		append(t1);append(t2);append(t3);} 
	list (T t1, T t2, T t3, T t4):list_v() {
		append(t1);append(t2);append(t3);append(t4);} 
	
	~list(void) {}

	// Copy and assignment.  Warning; these duplicate list structure,
	// so be a bit careful about calling them; they can be expensive.
	list(const list<T>& l):list_v(l) {}
	list<T>& operator=(const list<T>& l) {list_v::operator=(l); return *this;}
	
	// The following constructor creates a list from a length n array of T's
	//list(T *at, int  n);

	// MUTATORS

  	// Adding and removing things to lists	
	void append(const T t) { list_v::append((void*)t);} // Add t at the end of *this.
	void remove(const T t) { list_v::remove((void*)t);} //rem t from *this.
	// Tests...
	bool isEmpty() const { return (this->_headPtr == NULL ? true : false); }
	bool isNonEmpty() const {return (this->_headPtr == NULL ? false : true); }
	// Return true iff *this contains an element == to t.
	// Requires T::operator== or the like.
	#ifdef __GNUC__
	bool contains(const T const& t) const {return list_v::contains((const void*)t);}
	#else
	bool contains(const T & t) const {return list_v::contains((const void*)t);}
	#endif

/*
  int operator==(const list<T> l) const;
  int operator!=(const list<T> l) const{return !(*this == l);};

*/

	// Returns the position of 't' in the list, or -1 if not there
	// Requires T::operator== or similar (ok for pointers).
	// Note: totally un-debugged because I thought I needed it and then
	// came up with an easier solution to this problem
        #ifdef __GNUC__
	int positionOf(T const& t) const {return list_v::positionOf((const void*)t);}
        #else
	int positionOf(const T & t) const {return list_v::positionOf((const void*)t);}
        #endif
  
// Getting list_pos's to the ends of the list.

	// Get a list_pos (a marker to an element)
	// head and tail list_pos's allow modification of the list through 
	// the list_pos.
	// headConst and tailConst can be used to look along the list,
	// but not modify it.  They can be used in const contexts, though.
	
	inline list_pos<T> head() ;
	inline list_pos<T> headConst() const; 
	inline list_pos<T> tail() ;
	inline list_pos<T> tailConst() const;

// OPERATIONS: 
  
	const list<T>& operator+(const list<T>& lt) const {return (const list<T>&)list_v::operator+(lt);}	
	list<T>& operator+=(const list<T>& lt) {list_v::operator+=(lt); return *this;}	
	// list concatenation.  Copies both *this and lt, so it's not 
	// cheap.
	
	//void appendAndDestroy(list<T>& lt);
	// Append lt to *this.  Modifies *this (making it have lt at the end),
	// and sets lt to a null list.  This is quite cheap.
	// Hopefully, the alarming name will warn users that
	// it's got an unusual side effect.
	
	// toArray creates an array of T's from list. 
	//T *toArray() const;
	// returns n'th element
	// currently performs no error checking
	T nth_element(int n) const {return (T) list_v::nth_element(n);}
	T& tail_element(void) { return ((list_node<T>*)_tailPtr)->t();}
	T& head_element(void) { return ((list_node<T>*)_headPtr)->t();}
	
	//friend ostream& operator<<(ostream& o, const list<T>& l);

private: 
	list_v& _copy_me(void) const;
	
	friend class list_node<T>;
	friend class list_pos<T>;

	const list_node<T>* headPtr() const { return (list_node<T>*)_headPtr;}
	const list_node<T>* tailPtr() const { return (list_node<T>*)_tailPtr;}
	list_node<T>* headPtr() { return (list_node<T>*)_headPtr;}
	list_node<T>* tailPtr() { return (list_node<T>*)_tailPtr;}

	void create_first_element(const T t) { list_v::create_first_element((void*)t);}
	// UNSAFE.
	// Turns an empty list into a singleton containing t.
	// It does no checking, and will do bad things (probably
	// losing previous list contents) if the list is nonempty.

	list_node_v* _alloc_node(list_node_v *prev, void* t, 
							list_node_v *next) const {
		return alloc_node((list_node<T> *)prev,(T)t,(list_node<T> *)next);
	}
							
	list_node<T> * alloc_node(list_node<T> *prev, T t, list_node<T> *next) const;
	// Returns a new list_node<T> containing t, doing appropriate
	// memory allocation checks.  It doesn't really concern *this,
	// but we want to hide it inside the abstraction barrier, so it's
	// here.
};


// ------------------------------ list_pos ------------------------------
template<class T>
class list_pos: public list_pos_v {
  private:
	friend class list<T>;
	
	inline list_pos(list<T> *l, list_node<T>* n, bool consty);
	inline list_pos(list<T> *l, list_node<T>& n, bool consty);
	// Construct a list_pos to node n of list l, with
	// constant-ness consty.
	// Use list_pos<T> lp = l.head()   (or l.tail() if that's what you mean)
	// for internal use only.
		
	list_node<T>* cursor() {return (list_node<T>*)_cursor;}
	list<T>* tolist() const {return (list<T>*)_tolist;}
	list<T>* tolist() {return (list<T>*)_tolist;}
	//list_node<T> *_cursor;
	//list<T> *_tolist;
  public:

	// No ab nihilo constructors.
	// The ordinary way of making a list_pos on a list is,
	// list_pos<T> lp = l.head();
	// (or l.tail() if you prefer);
	// or one of the other list_pos-returning functions when and if
	// they are added.
	//
	// Basically, we consider "list_pos<T> lp(l)" to be
	// ambiguous about which end of l lp will point to.
	// However, I don't, so I've added the constructor to the head();
	
	// Assignment and copying is normal.
	list_pos(const list_pos<T>& lp):list_pos_v(lp) {}
	list_pos<T>& operator=(const list_pos<T>& l) {set_equal(l); return *this;}
	list_pos(list<T>& l):list_pos_v(l) {}
	list_pos(const list<T>& l):list_pos_v(l) {}
	
	// Operations
	bool operator<(const list_pos<T>& b) const {return list_pos_v::operator<(b);}	
	bool operator==(const list_pos<T>& b) const {return list_pos_v::operator==(b);}

	// Destroying the listpos doesn't do anything to the list
	~list_pos(void) {}

	// get the element at this position.
	// We provide two names because (A) each made sense in
	// some phrasing, and (B) we couldn't think of a good
	// single name when we wrote that part of the code.
	// Bad style, really.
	// These functions blow up if the list_pos is offlist.
	T here() const { return (T) list_pos_v::here_v();}
	T item() const { return this->here(); };

	// Get back to the list that this list_pos points to.  
	list<T>* mylist() const {return this->tolist();};
	list<T>* mylist() {return this->tolist();};
  
// MUTATING THE LIST ITSELF

	// Insert an element t after the current position.
	// Leave this list_pos pointed at the new element.
	// I'm not sure what the right behavior should be when
	// *this is off the list; so it causes an error.
	// However, when the list is empty, this adds a single element to it.
	void insertNext(T t) { list_pos_v::insertNext((void*)t);}
	
	// Insert t before the current position; leave *this
	// pointing to new position.
	// Like insertNext, mutatis mutandis.
	void insertPrev(T t) { list_pos_v::insertPrev((void*)t);}
		
	// replace referenced list element with t.
	// Doom, if the list_pos doesn't refer to an element
	// of a non-constant list.
	void replaceWith(T t) { list_pos_v::replaceWith((void*)t);}
		
	// Returns a list consisting of this through b.
	//list<T> through(const list_pos<T> b) const;

  
};  

#include "list.icc"

#endif

  
