/*
 * @(#)TimedProcess.java
 *
 * Copyright (C) 2001,2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.util.thread.v1;


/**
 * A Utility to run a task in a thread, and to kill it if the thread runs
 * too long.  This will first attempt an interrupt() on the thread.  If that
 * doesn't work, then a stop() will be executed.  This is a well-documented
 * bad practice, so you have been warned!
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2003/02/10 22:52:49 $
 * @since     15-Mar-2002
 */
public class TimedProcess
{
    protected static TimedProcess s_instance = new TimedProcess();
    
    public static final int REASONABLE_TIME = 500;
    
    /**
     * Implement this interface to have your own method for killing the
     * runner and thread.  The caller will perform the Thread.join() manually.
     * If the thread is still running after every possible attempt to stop
     * the thread has occured (run this interface, and join), then a stop()
     * will be called on the thread.
     */
    public static interface RunnableKiller
    {
        public void killRunnable( Runnable r, Thread t );
    }
    
    
    private static class InterruptRunnableKiller implements RunnableKiller
    {
        public void killRunnable( Runnable r, Thread t )
        {
            t.interrupt();
        }
    }
    
    private RunnableKiller interruptor = new InterruptRunnableKiller();
    
    
    
    protected TimedProcess()
    {
        // do nothing
    }
    
    
    public static TimedProcess getInstance()
    {
        return s_instance;
    }
    
    
    /**
     * Runs the given process in a Thread.  If it does not stop within the
     * specified time, then attempts will be made to kill the thread
     * (see earlier text).  If the process did not complete in the specified
     * time, an <tt>InterruptedException</tt> will be thrown.
     *
     * @exception InterruptedException if the process did not complete within
     *      the given time frame.
     */
    public void runTimed( Runnable process, long waitTimeMillis )
            throws InterruptedException
    {
        runTimed( process, waitTimeMillis, this.interruptor );
    }
    
    
    /**
     * Runs the given process in a Thread.  If it does not stop within the
     * specified time, then attempts will be made to kill the thread
     * (see earlier text).  If the process did not complete in the specified
     * time, an <tt>InterruptedException</tt> will be thrown.
     *
     * @exception InterruptedException if the process did not complete within
     *      the given time frame.
     */
    public void runTimed( Runnable process, long waitTimeMillis,
            RunnableKiller processEnder )
            throws InterruptedException
    {
        if (processEnder == null || process == null ||
            waitTimeMillis <= 0L)
        {
            throw new IllegalArgumentException("no null args");
        }
        
        Thread t = new Thread( process, "TimedProcess" );
        setupThread( t );
        t.start();
        joinThread( t, waitTimeMillis );
        if (t.isAlive())
        {
            stopThread( t, process, processEnder );
            
            throw new InterruptedException( "Process did not end within "+
                waitTimeMillis+" milliseconds." );
        }
    }
    
    
    
    /**
     * 
     */
    protected void setupThread( Thread t )
    {
        // do nothing
    }
    
    
    /**
     * 
     */
    protected void stopThread( Thread t, Runnable process,
            RunnableKiller processEnder )
    {
        processEnder.killRunnable( process, t );
        joinThread( t, REASONABLE_TIME );
        if (t.isAlive())
        {
            // We'll give it one more shot before we really kill it.
            t.interrupt();
            joinThread( t, REASONABLE_TIME );
            if (t.isAlive())
            {
                // ok, that's enough.
                // Let's do this the hard way.
                t.stop();  // we know this is bad, but do it anyway
                if (t.isAlive())
                {
                    throw new IllegalStateException(
                        "After stopping the thread, the thread still lives.  "+
                        "The thread cannot be killed." );
                }
            }
        }
    }
    
    
    protected void joinThread( Thread t, long joinTime )
    {
        try
        {
            t.join( joinTime );
        }
        catch (InterruptedException e)
        {
            // ignore
        }
    }
}

